#include "symtab.h"
#include "dtype.h"
#include "symbol.h"
#include "ieeesymtab.h"
#include "core.h"
#include "ieee.h"
SRCFILE("ieeesymtab.c")

#define	TYP_START	256	/* Starting index of per file type info */

int ccdemangle(char**,char* =0,int =0);

IeeeSymTab::IeeeSymTab(class Core* c,int fd,SymTab *s,long r):SymTab(c,fd,s,r)
{
}

IeeeSymTab::~IeeeSymTab()
{
}

char *IeeeSymTab::gethdr()
{
	int i;
	char *err;

	if (err = readblock(0L, 90))
		return err;
	if (lgetc() != CMD_MB) {
		err = "IeeeSymTab::gethdr: not a IEEE file";
		goto out;
	}
	skipentry();
	if (lgetc() != CMD_AD) {
		err = "IeeeSymTab::gethdr: Address Descriptor missing";
		goto out;
	}
	skipentry();
	for(i = 0; i < IEEE_NOFFSETS; i++) {
		if (lgetc() != CMD_AS || lgetc() != STVAR_W || getint() != i) {
			err = "IeeeSymTab::gethdr: bad AS W[i]";
			goto out;
		}
		offsets[i] = getint();
	}
	entries = 1;
out:
	freeblock();
	return err;
}

Source *IeeeSymTab::tree()
{
	IeeeSource	*src = 0;
	Var	*glb;
	int	blktype;
	unsigned char	*typestart, *typeend;
	unsigned char	*dclrstart, *dclrend;
	unsigned char	*linestart, *lineend;

	glb = globregs(_blk, _core->nregs());
	if (_warn = readblock(offsets[4], (int)(offsets[5] - offsets[4])))
		return 0;
	while (fcur < fend) {
		typestart = fcur;
		if (lgetc() != CMD_BS)
			break;
		blktype = getint();
		typeend = typestart + getint();
		fcur = typeend;
		if (blktype != BLK_TYPES)
			continue;
		if (lgetc() != CMD_BS || getint() != BLK_DECL)
			break;
		dclrstart = typeend;
		dclrend = dclrstart + getint();
		fcur = dclrend;
		if (lgetc() != CMD_BS || getint() != BLK_LINFO)
			break;
		linestart = dclrend;
		lineend = linestart + getint();

		fcur = linestart;
		if (lineinfo(&src)) {
			fcur = typestart;
			typeinfo(src);
			fcur = dclrstart;
			funcinfo(src, &glb);
		}

		fcur = lineend;
	}
	freeblock();
	while (src && src->lsib)
		src = (IeeeSource*)src->lsib;
	globals(&glb);
	return src;
}

int IeeeSymTab::lineinfo(IeeeSource **src)
{
	int c, index;
	int	lcnt = 0;
	int	l, lastl;
	long	a, lasta;

	if (lgetc() != CMD_BS || getint() != BLK_LINFO)
		return 0;
	getint();
	char *fname = getstring();
	unsigned char *rewind = fcur;
	lastl = 0;
	lasta = 0;
	while ((c = lgetc()) != CMD_BE) {
		switch(c) {
		case CMD_AT:
			if (lgetc() != STVAR_N)
				return 0;
			index = getint();
			skipint(2);
			l = getint();
			skipint(1);
			if (lgetc() != CMD_AS || lgetc() != STVAR_N ||
		    	    getint() != index)
				return 0;
			a = getint();
			if (a != lasta && lastl != l)
				lcnt++;
			lasta = a;
			lastl = l;
			break;
		case CMD_BS:
			skipblock();
			break;
		case CMD_NN:
			break;
		default:
			return 0;
		}
		skipentry();
	}
	if (!lcnt)
		return 0;
	*src = new IeeeSource(this, *src, sf("%s", fname), 0);
	IeeeLineEntry *lp = new IeeeLineEntry[lcnt];
	(*src)->lineinfo = lp;
	(*src)->linecnt = lcnt;
	fcur = rewind;
	lastl = 0;
	lasta = 0;
	for (IeeeLineEntry *le = lp + lcnt; lp < le; skipentry()) {
		if ((c = lgetc()) == CMD_AT) { // AT N indx 0 7 lineno code
			lgetc(); skipint(3);
			l = getint();
			skipint(1);
			lgetc(); lgetc(); skipint(1);	// AS N indx address
			a = getint();
			if (a == lasta)
				lp[-1].lineno = l;
			else if (l != lastl) {
				lp->lineno = l;
				lp->addr = a;
				lp++;
			}
			lasta = a;
			lastl = l;
		} else if (c == CMD_BS)
			skipblock();
	}
	return 1;
}

void IeeeSymTab::typeinfo(IeeeSource *src)
{
	char	*id;
	int	nindx;
	int	indx, maxindx = 0;
	int	i, c;
	DType	*dp;
	UType	*u;
	int	ssize;
	unsigned char *save;

	if (lgetc() != CMD_BS || getint() != BLK_TYPES)
		return;
	skipentry();
	save = fcur;
	// 1st pass, compute the size of typeinfo array
	while ((c = lgetc()) != CMD_BE) {
		if (c != CMD_NN)
			return;
		skipentry();
		if (lgetc() != CMD_TY)
			return;
		indx = getint();
		if (indx > maxindx)
			maxindx = indx;
		skipentry();
	}
	if (!maxindx)
		return;
	src->typecnt = maxindx - TYP_START + 1;
	src->typeinfo = new DType[src->typecnt];
	
	// 2nd pass, fill it in
	fcur = save;
	while ((c = lgetc()) != CMD_BE) {
		nindx = getint();
		id = getstring();
		lgetc();	// CMD_TY
		indx = getint();
		if (lgetc() != STVAR_N || getint() != nindx) {
			skipentry();
			continue;
		}
		dp = &src->typeinfo[indx - TYP_START];
		switch(c = lgetc()) {
		case 'P':		// Pointer
			dp->pcc = PTR;
			dp->univ = new DType;
			*(dp->ref()) = src->gettype(getint());
			break;
		case 'Z':		// Array
			dp->pcc = ARY;
			dp->univ = new DType;
			*(dp->ref()) = src->gettype(getint());
			dp->dim = getint() + 1;
			break;
		case 'N':		// Enum
		case 'S':		// Structure
		case 'U':		// Union
			if (c == 'N')
				ssize = 4;
			else
				ssize = getint();
			save = fcur;
			skipentry();
			if (!id[0])
				id = sf("%s.%d", src->text(), src->sindex++);
			else
				id = ccdemangle(&id) ? id : sf("%s", id);
			if (u = (UType *)idtosym(U_UTYPE, id, 0))
				break;
			++UTypeStubs;
			u = new UType(this, offsets[4] + save - fstart,
					fcur - save, id);
			u->range.lo = ssize;
			u->src = src;
			if (c == 'S')
				u->type.pcc = STRTY;
			else if (c == 'U')
				u->type.pcc = UNIONTY;
			else
				u->type.pcc = ENUMTY;
			u->type.univ = u;
			u->rsib = utype;
			utype = u;
			dp->univ = u;
			dp->pcc = u->type.pcc;
			break;
		case 'T':		// Typedef
			*dp = src->gettype(getint());
			break;
		case 'V':		// void
			dp->pcc = UNDEF;
			break;
		case 'x':		// Function
			skipint(3);
			dp->pcc = FTN;
			dp->univ = new DType;
			*(dp->ref()) = src->gettype(getint());
			break;
		case 'g':		// Field
			i = getint();
			if (i == 0)
				dp->pcc = UBITS;
			else if (i == 1)
				dp->pcc = UBITS;
			else
				break;
			dp->dim = getint();
			break;
		default:
			break;
		}
		skipentry();
	}
}

void IeeeSymTab::funcinfo(IeeeSource *src, Var **glb)
{
	int	c;
	char	*id;
	int	type;
	int	stclass;
	int	nindex;
	long	addr, eaddr;
	Var	*v, *fst = 0;
	Func	*func = 0;
	unsigned char *save;
	IeeeLineEntry *lf, *ll, *le;
	char	*name;
	int	demangled;

	if (lgetc() != CMD_BS || getint() != BLK_DECL)
		return;
	skipentry();
	if (lgetc() != CMD_NN)
		return;
	skipentry();
	if (lgetc() != CMD_AT)
		return;
	skipentry();
	while ((c = lgetc()) == CMD_AS)
		skipentry();
	if (c != CMD_AT)
		return;
	skipentry();
	while ((c = lgetc()) != CMD_BE) {
	  switch(c) {
	    case CMD_BS:
		if (getint() != BLK_FUNC)
			return;
		getint();	// Size of block
		name = id = getstring();
		demangled = ccdemangle(&name);
		getint();	// Frame size
		type = getint();
		addr = getint();
		save = fcur;
		skipblock();
		eaddr = getint() + 1;
		le = &src->lineinfo[src->linecnt];
		for (lf = src->lineinfo; lf < le && lf->addr < addr; lf++)
			;
		for (ll = lf; ll < le && ll->addr < eaddr; ll++)
			;
		if (lf == ll) // No line info implies from include
			break;
		++FunctionStubs;
		if (!demangled)
			name = sf("%s", name);
		func = new Func(this, name, src, lf->lineno);
		if (demangled) {
			name = id;
			ccdemangle(&name, 0, 1);
			func->namewithargs = name;
		}	
		func->begin = offsets[4]+save-fstart;
		func->size = fcur - save;
		func->range.lo = addr;
		func->range.hi = eaddr;
		func->type = src->gettype(type);
		func->regsave = -1;
		func->lnnoptr = lf - src->lineinfo;
		func->lines.hi = ll[-1].lineno;
		break;
	    case CMD_NN:
		nindex = getint();
		id = getstring();
		if (lgetc() != CMD_AT || lgetc() != STVAR_N || getint()!=nindex)
			break;
		type = getint();
		stclass = getint();
		if (stclass != 5 && stclass != 8 && stclass != 3)
			break;
		if (stclass == 5)
			addr = 0;
		else {
			if (lgetc() != CMD_AS || lgetc() != STVAR_N ||
				getint()!= nindex)
				break;
			addr = getint();
		}
		demangled = ccdemangle(&id);
		if (!demangled)
			id = sf("%s", id);
		if (stclass == 3)
			gathervar(id, addr, &fst, src->blk, U_FST, src, type);
		else {
			if (v = (Var*)idtosym(U_GLB, id, 0)) {
				if (addr && !v->range.lo)
					v->range.lo = addr;
				break;
			}
			gathervar(id, addr, glb, _blk, U_GLB, src, type);
		}
		break;
	  }
	  skipentry();
	}
}

char *IeeeSymTab::globals(Var **glb)
{
	Block	*fake = fakeblk();
	Func	*func = 0;
	Var	*v;
	DType	*d;
	char	*err;
	int	index, c, i, addr, type;
	char	*id;
	int	demangled;

	if (err = readblock(offsets[3], (int)(offsets[4] - offsets[3])))
		return err;
	while (fcur < fend) {
	 switch(c = lgetc()) {
		case CMD_NI:
			index = getint();
			id = getstring();
			break;
		case CMD_AT:
			c = lgetc();
			i = getint();
			if (c == STVAR_I)
				type = getint();
			skipentry();
			break;
		case CMD_AS:
			if (lgetc() != STVAR_I) {
				err = "IeeeSymTab::globals: AS not I";
				goto out;
			}
			i = getint();
			if (i != index) {
				err = "IeeeSymTab::globals: wrong index";
				goto out;
			}
			addr = getint() + (int)relocation;
			if (*id == '_')
				id++;
			demangled = ccdemangle(&id);
			if (type == 0xF) {
				if (idtosym(U_FUNC, id, 0))
					break;
				if (!demangled)
					id = sf("%s", id);
				func = new Func(this,id,0,0);
				func->range.lo = addr;
				func->_blk = fake;
				d = new DType;
				d->pcc = INT;
				func->type = d->incref();
				func->type.pcc = FTN;
			} else {
				if (v = (Var*)idtosym(U_GLB, id, 0)) {
					if (!v->range.lo)
						v->range.lo = addr;
					break;
				}
				if (!demangled)
					id = sf("%s", id);
				gathervar(id, addr, glb, _blk, U_GLB);
			}
			break;
		case CMD_NN:
			getint();
			getstring();
			break;
		default:
			err = "IeeeSymTab::globals: unexpected command";
			goto out;
	 }
	}
out:
	freeblock();
	return err;
}

void IeeeSymTab::gathervar(char *id, long addr, Var **v, Block *b, UDisc d,
			IeeeSource *src, int type)
{
	*v = new Var(this, b, *v, d, id);
	if (b && !b->var)
		b->var = *v;
	if (src)
		(*v)->type = src->gettype(type);
	else
		(*v)->type.pcc = INT;
	(*v)->range.lo = addr;
	if (d == U_GLB || d == U_STA || d == U_FST)
		(*v)->range.lo += relocation;
}

Block *IeeeSymTab::gatherfunc(Func *func)
{
	Var *arg = 0, *lcl = 0;
	Stmt *stmt = 0;
	int nest;
	int o, t, nindex, stclass;
	char *id;

	++FunctionGathered;
	Block *ablk = new Block(this, 0, 0, sf("%s().arg_blk", func->text()));
	Block *lblk = new Block(this, ablk, 0, sf("%s().lcl_blk",func->text()));
	ablk->child = lblk;
	IeeeSource *src = (IeeeSource *)func->source();
	readblock(func->begin, (int)func->size);
	
	for (nest = 1; nest; skipentry()) {
	switch(lgetc()) {
	case CMD_BS:
		nest++;
		break;
	case CMD_BE:
		nest--;
		break;
	case CMD_NN:
		nindex = getint();
		id = getstring();
		if (lgetc() != CMD_AT || lgetc() != STVAR_N || getint()!=nindex)
			break;
		id = sf("%s", id);
		t = getint();
		switch(stclass = getint()) {
		case 1:		// Argument, local
			if ((o = getint()) < 0)
				gathervar(id, o, &lcl, lblk, U_AUT, src, t);
			else
				gathervar(id, o, &arg, ablk, U_ARG, src, t);
			break;
		case 2:		// Register variable
			if ((o = getint()) == 0 || o == 8)
				break;
			gathervar(id, o, &lcl, lblk, U_REG, src, t);
			break;
		case 3:		// Function static
			if (lgetc() != CMD_AS || lgetc() != STVAR_N ||
				getint() != nindex)
					break;
			gathervar(id, getint(), &lcl, lblk, U_STA, src, t);
			break;
		case 10:	// Register variable
			gathervar(id, getint(), &lcl, lblk, U_REG, src, t);
			o = (lgetc() & 0xE0) == 0xE0;
			lungetc();
			if (!o)	// Register arg
				gathervar(id,getint(),&arg,ablk,U_ARG,src,t);
			break;
		}
		break;
	}
	}
	freeblock();

	// Line info
	IeeeLineEntry *lp, *le;
	le = &src->lineinfo[src->linecnt];
	lp = &src->lineinfo[func->lnnoptr];
	while (lp < le && lp->addr < func->range.hi) {
		if (stmt)
			stmt->range.hi = lp->addr;
		stmt = new Stmt(this, lblk, stmt);
		if (!ablk->stmt)
			ablk->stmt = stmt;
		stmt->lineno = lp->lineno;
		stmt->range.lo = lp->addr;
		ablk->range.hi = lp->addr;
		lp++;
	}
	if (stmt)
		stmt->range.hi = func->range.hi;

	uncfront(ablk->var, (char *)0);
	uncfront(lblk->var, (char *)0);
	return ablk;
}

Var *IeeeSymTab::gatherutype(UType *u)
{
	Var *first = 0, *v = 0;
	char *id;
	int isenum = u->type.pcc == ENUMTY;

	++UTypeGathered;
	IeeeSource *src = (IeeeSource *)u->src;
	readblock(u->begin, (int)u->size);
	while (fcur < fend) {
		id = sf("%s", getstring());
		v = new Var(this, 0, v, U_MOT, id);
		if (isenum) {
			v->range.lo = getint();
			v->type.pcc = MOETY;
		} else {
			v->type = src->gettype(getint());
			v->range.lo = getint();
			int type = v->type.pcc;
			if (type == BITS || type == UBITS)
				v->range.lo = (v->range.lo & ~0x1f) + 32 -
					v->type.dim - (v->range.lo & 0x1f);
		}
		if (!first)
			first = v;
	}
	if (!isenum)
		uncfront(first, u->_text);
	freeblock();
	return first;
}

char *IeeeSymTab::readblock(long offset, int size)
{
	fstart = new unsigned char[size];
	if (lseek( fd, offset, 0) == -1  ||
	    !ReadOK(fd, (char*)fstart, size)) {
		delete [] fstart;
		return "IeeeSymTab::readblock: failed";
	}
	fend = &fstart[size];
	fcur = fstart;
	return 0;
	
}

void IeeeSymTab::freeblock()	{ delete [] fstart; }
void IeeeSymTab::lungetc()	{ fcur--; }
void IeeeSymTab::skipint(int i)	{ while(i--) getint(); }

int IeeeSymTab::lgetc()
{
	if (fcur >= fend)
		return -1;
	else
		return (int)*fcur++;
}

void IeeeSymTab::skipentry()
{
	for(;;) {
		int c = lgetc();
		if (c == -1)
			return;
		if ((c & 0xE0) == 0xE0) {
			lungetc();
			return;
		}
		if ((c & 0xF0) == 0x80) {
			lungetc();
			getint();
		}
	}
}

void IeeeSymTab::skipblock()
{
	int nest = 1;
	int c;

	for(skipentry(); ;skipentry()) {
		if ((c = lgetc()) == CMD_BS)
			nest++;
		else if (c == CMD_BE && --nest <= 0)
			break;
	}
}

int IeeeSymTab::getint()
{
	int c;
	int cnt;
	int val = 0;

	if ((c = lgetc()) & 0x80) {
		if (c != 0x80) {
			cnt = c & 0x0F;
			while (cnt > 4) {
				lgetc();
				cnt--;
			}
			while (cnt--)
				val = (val << 8) | lgetc();
		} else
			val = 0;
	} else
		val = c;
	return val;
}

char * IeeeSymTab::getstring()
{
	static char s[128];
	char *cp = s;
	int cnt = lgetc() & 0x7F;

	while(cnt--)
		*cp++ = lgetc();
	*cp = 0;
	return s;
}

IeeeSource::IeeeSource(SymTab *stab, Source *left, char *t, long c)
 :Source(stab, left, t, c)	{}

IeeeSource::~IeeeSource()
{
	DType *d, *de;

	delete [] lineinfo;
	for (d = typeinfo, de = typeinfo + typecnt; d < de; d++)
		d->free();
	delete [] typeinfo;
}

DType IeeeSource::gettype(int i)
{
	DType d;

	if (i < TYP_START) {
		switch(i) {
			case 3:		d.pcc = UCHAR; break;
			case 5:		d.pcc = USHORT; break;
			case 7:		d.pcc = ULONG; break;
			case 10:	d.pcc = FLOAT; break;
			case 11:	d.pcc = DOUBLE; break;
			case 16:	d.pcc = INT; break;
			case 18:	d.pcc = UNSIGNED; break;
			case 19:	d.pcc = CHAR; break;
			case 20:	d.pcc = LONG; break;
			case 23:	d.pcc = SHORT; break;
		}
	} else {
		i -= TYP_START;
		if (i < typecnt)
			d = chain(&typeinfo[i]);
	}
	return d;
}

DType IeeeSource::chain(DType *dp)
{
	DType d;

	d.pcc = dp->pcc;
	d.dim = dp->dim;
	d.univ = dp->univ;
	if (d.pcc & TMASK) {
		d.univ = new DType;
		*(d.ref()) = chain(dp->ref());
	}
	return d;
}
